﻿using System;

namespace MicroRWD.Common
{
    // Handles the management of a single port
    public class PortHandler : IDisposable
    {
        #region Private Constants

        // The maximum length of time to wait for CTS (in milliseconds)
        private const int C_CTS_TIMEOUT_MS = 3000;
        
        #endregion


        #region Private Properties

        // The underlying port
        private Port Port { get; set; }

        // Retries Counter
        private int Retries = 1;

        #endregion


        #region Public Properties

        // The port name
        public string PortName { get { return Port.PortName; } }

        // The port state
        public PortState PortState { get; private set; }

        #endregion


        #region Constructor

        // Constructs a new port handler using the specified port name
        public PortHandler(string _portName)
        {
            // Create a port instance to communicate with the serial port
            Port = new Port(_portName);

            // Establish the initial port state
            if (!Port.IsOpen)
            {
                // Not open
                PortState = PortState.Closed;
            }
            else
            {
                // Open but not connected
                PortState = PortState.Open;
            }
        }

        #endregion


        #region Public Methods

        // Send command to MicroRWD via associated port and wait for reply with timeout (in milliseconds)
        public byte[] Request(byte[] _command, int _commandTimeoutMillis)
        {
            // Zero length reply on timeout, data otherwise
            byte[] result = new byte[0];

            // Don't poll unless connected
            if (Port.IsOpen)
            {
                // Wait for CTS line to be asserted
                if (Port.WaitForCTS(C_CTS_TIMEOUT_MS))
                {
                    // Send command
                    Port.SendCmd(_command);
                    Retries = 1;

                    // Wait for reply (or timeout)
                    result = Port.GetReply(_commandTimeoutMillis);

                    // Retry if timeout (i.e. no response)
                    int extendedTimeout = _commandTimeoutMillis;
                    while ((result.Length == 0) && (Retries != 0))
                    {
                        // Wait for reply (or timeout) again in case things are moving slower than expected
                        result = Port.GetReply(_commandTimeoutMillis);

                        // Retry again if that didn't work
                        if (result.Length == 0)
                        {
                            extendedTimeout = extendedTimeout > 8000 ? extendedTimeout : extendedTimeout * 2; // double the time out each time through

                            // Send command
                            Port.SendCmd(_command);

                            // Wait for reply (or timeout)
                            result = Port.GetReply(extendedTimeout);

                            --Retries;
                        }
                    }

                    // Check for response so we can maintain current port state
                    if (result.Length > 0)
                    {
                        // Connected
                        PortState = PortState.Connected;
                    }
                    else
                    {
                        // Open but not connected
                        PortState = PortState.Open;

                        // Information
                        Log.Information("timeout waiting for response");
                    }
                }
                else
                {
                    // Open but not connected
                    PortState = PortState.Open;

                    // Information
                    Log.Information("timeout waiting for CTS signal");
                }
            }
            else
            {
                // Not open
                PortState = PortState.Closed;

                // Warning
                Log.Warning("port is not open");
            }

            // Return result
            return result;
        }

        #region IDisposable Methods

        // Dispose of the underlying resources
        public void Dispose()
        {
            // Dispose of the resources allocated by the port
            Port.Dispose();

            // Prevent further finalisation
            GC.SuppressFinalize(this);
        }

        #endregion

        #endregion
    }
}
